from pyparsing import Optional, Literal, Word, printables, nums, OneOrMore

from codeable_detectors.basic_detectors import AtLeastOneFileMatchesDetector
from codeable_detectors.detector_context import DetectorContext
from codeable_detectors.evidences import FailedEvidence, Evidence, ComponentEvidence
from codeable_detectors.match import update_keyword_args_on_matches
from codeable_detectors.utils import update_keyword_args


def detect_docker_from_statements(ctx, name, version=None):
    if version is None:
        # as version can be "latest" or omitted put into an ARG variable, we don't specify 
        # it in detail here (from can also define an alias [AS <name>]; omitted for now as well)

        # num = Word(nums)
        # version_pattern = num + "." + num + "." + num
        version_pattern = Optional(Literal(":") + Word(printables))
    else:
        # if the user specifies it, it should appear as specified
        version_pattern = Literal(":") + Literal(version)
    pattern = Literal("FROM") + Literal(name) + version_pattern

    matches = []
    docker_from_matches = ctx.matches_pattern(pattern)
    if docker_from_matches:
        update_keyword_args_on_matches(docker_from_matches, name=name)
        matches.append(docker_from_matches)
    return matches


def detect_docker_expose_statements(ctx, port=None):
    if port is None:
        port_pattern = Word(nums)
    else:
        port_pattern = Literal(port)
    pattern = Literal("EXPOSE") + port_pattern

    matches = []
    docker_expose_matches = ctx.matches_pattern(pattern)
    for match in docker_expose_matches:
        match.update_keyword_args(port=match.text[6:].strip())
        matches.append(match)
    return matches


class DockerEnvStatements(AtLeastOneFileMatchesDetector):
    def __init__(self):
        super().__init__()
        self.file_names = ["Dockerfile"]
        self.docker_env_pattern = Literal("ENV") + OneOrMore(Word(printables, excludeChars="=\\") +
                                                             Literal("=") + Word(printables,
                                                                                 excludeChars="=\\") + Optional("\\"))
        self.env_lines_pattern = (Word(printables, excludeChars="=\\") +
                                  Literal("=") + Word(printables, excludeChars="=\\"))

    def detect_in_context(self, ctx, **kwargs):
        matches = []
        docker_env_matches = ctx.matches_pattern(self.docker_env_pattern)
        for docker_env_match in docker_env_matches:
            print("ENV MATCH = " + docker_env_match.text)
            env_lines_matches = DetectorContext(docker_env_match).matches_pattern(self.env_lines_pattern)
            if env_lines_matches:
                matches.append(docker_env_match)
                for envLinesMatch in env_lines_matches:
                    print("LINE MATCH = " + envLinesMatch.text)
                    line_parts = envLinesMatch.text.split("=")
                    docker_env_match.kwargs[line_parts[0].strip()] = line_parts[1].strip()
        if not matches:
            return FailedEvidence("Docker ENV statement not found")
        return Evidence(matches)


class DockerImageBasedComponent(AtLeastOneFileMatchesDetector):
    def __init__(self, base_image_name, **kwargs):
        super().__init__()
        self.file_names = ["Dockerfile"]
        self.base_image_name = base_image_name
        options = update_keyword_args({'is_exposed': None,
                                       'exposed_port': None,
                                       'base_image_version': None}, kwargs)
        self.is_exposed = options['is_exposed']
        self.exposed_port = options['exposed_port']
        self.base_image_version = options['base_image_version']

    def detect_in_context(self, ctx, **kwargs):
        matches = []

        base_image_matches = detect_docker_from_statements(ctx, self.base_image_name, self.base_image_version)
        if not base_image_matches:
            return FailedEvidence("docker FROM statement for base image '" + self.base_image_name + "' not found")

        if base_image_matches:
            matches = base_image_matches
            if self.is_exposed:
                expose_matches = detect_docker_expose_statements(ctx, self.exposed_port)
                if not expose_matches:
                    FailedEvidence("docker EXPOSE statement not found")
                matches.extend(expose_matches)

        if not matches:
            return FailedEvidence("dockerized base image '" + self.base_image_name + "' detection failed")

        return ComponentEvidence(matches, technology_types=["dockerized"])


class DockerMySQLDBImageBasedComponent(DockerImageBasedComponent):
    def __init__(self):
        super().__init__("mysql")

    def detect_in_context(self, ctx, **kwargs):
        evidence = super().detect_in_context(ctx, **kwargs)
        if not evidence.has_succeeded():
            return evidence
        return evidence.set_properties(detector_name="MySQL DB",
                                       detector_component_types="mySQLDB", detector_link_types="mySQLProtocol",
                                       detector_technology_types="mysql", kwargs=kwargs)
